<?php

/**
 * Represents a contact address object in OpenEMR and in the database.  Follows the Active Record design pattern for
 * loading and persisting data to the database.
 *
 * @package   OpenEMR
 * @link      https://www.open-emr.org
 *
 * @author    David Eschelbacher <psoas@tampabay.rr.com>
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Copyright (c) 2022 David Eschelbacher <psoas@tampabay.rr.com>
 * @copyright Copyright (c) 2022 Discover and Change, Inc. <snielson@discoverandchange.com>
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Common\ORDataObject;

use OpenEMR\Common\ORDataObject\Contact;
use OpenEMR\Common\ORDataObject\Address;
use DateTime;
use OpenEMR\Services\Utils\DateFormatterUtils;

class ContactAddress extends ORDataObject implements \JsonSerializable
{
    // ORDataObject won't save with a numeric "0", we could change the condition but we don't know the ramifications
    // of doing that and we're short on time.  So we are switching this to string constants and letting MySQL convert it
    // TODO: Change these to numeric values once we've fixed the issue in ORDataObject
    const STATUS_ACTIVE = 'A';
    const STATUS_INACTIVE = 'I';
    const IS_PRIMARY_YES = "Y";
    const IS_PRIMARY_NO = "N";
    const ADDRESS_TYPE_HOME = "home";
    const DEFAULT_TYPE = "both"; // both physical and shipping address
    const DEFAULT_USE = self::ADDRESS_TYPE_HOME;
    const USE_OLD = "old"; // option_id for 'use' property when the address is no longer used or is incorrect.
    private $id;
    /**
     * @var int The foreign key to the contact table
     */
    private $contact_id;

    /**
     * @var int The foreign key id to the address table
     */
    private $address_id;

    /**
     * @var int The priority of this address for the given contact
     */
    private $priority;

    /**
     * @var string The type of address this is (physical, postal, etc)
     */
    private $type;

    /**
     * @var Note information about the address
     */
    private $notes;

    /**
     * @var bool Whether the address is active or not
     */
    private $status;

    /**
     * @var string Whether this is the primary / default address for the contact
     */
    private $isPrimary;

    /**
     * @var Datetime The date this address was created at
     */
    private $createdDate;

    /**
     * @var Datetime The start date for this address
     */
    private $periodStart;

    /**
     * @var Datetime The end date for this address
     */
    private $periodEnd;

    /**
     * @var string The username of the user that created this address
     */
    private $author;

    /**
     * @var string The explanation for why the address was inactivated.
     */
    private $inactivated_reason;

    /**
     * @var Contact The object pointer to the contact object.  Only kept in memory
     */
    private $_contact;

    /**
     * @var Address The object pointer to the address object.  Only kept in memory.
     */
    private $_address;

    /**
     * @var string The use of this address (home, work, etc)
     */
    private $_use;

    /**
     * Constructor sets all Address attributes to their default value
     */
    public function __construct($id = "")
    {
        parent::__construct("contact_address");
        $this->setThrowExceptionOnError(true);
        // we set our defaults, populate can override this if  needed.
        $this->id = $id;
        $this->priority = 0;
        $this->author = $_SESSION['authUser'];
        $this->status = self::STATUS_ACTIVE;
        $this->use = self::DEFAULT_USE;
        $this->type = self::DEFAULT_TYPE;
        $this->isPrimary = false;
        $this->createdDate = new DateTime();
        $this->periodStart = $this->createdDate;

        if ($id != "") {
            $this->populate();
            $this->setIsObjectModified(false);
        }
    }

    protected function get_date_fields()
    {
        return ['created_date', 'period_start', 'period_end'];
    }

    public function populate_array($results)
    {
        if (is_array($results)) {
            foreach ($this->get_date_fields() as $field) {
                if (isset($results[$field])) {
                    $results[$field] = \DateTime::createFromFormat("Y-m-d H:i:s", $results[$field]);
                }
            }
        }
        parent::populate_array($results); // TODO: Change the autogenerated stub
    }

    public function persist()
    {
        // first we will persist our address
        if ($this->getAddress()->isObjectModified()) {
            $this->getAddress()->persist();
            $this->set_address_id($this->getAddress()->get_id());
        }
        if ($this->getContact()->isObjectModified()) {
            $this->getContact()->persist();
            $this->set_contact_id($this->getContact()->get_id());
        }

        // now we can persist with our objects saved off
        return parent::persist(); // TODO: Change the autogenerated stub
    }

    public function setAddress(Address $address)
    {
        $this->_address = $address;
        $this->address_id = $address->get_id();
        $this->setIsObjectModified(true);
    }

    public function setContact(Contact $contact)
    {
        $this->_contact = $contact;
        $this->contact_id = $contact->get_id();
        $this->setIsObjectModified(true);
    }

    public function getContact(): Contact
    {
        if (empty($this->_contact)) {
            $this->_contact = $this->loadContact($this->contact_id);
        }
        return $this->_contact;
    }

    public function getAddress(): Address
    {
        if (empty($this->_address)) {
            $this->_address = $this->loadAddress($this->address_id);
        }
        return $this->_address;
    }

    public function deactivate()
    {
        $this->periodEnd = new DateTime();
        $this->set_status(self::STATUS_INACTIVE);
        $this->setIsObjectModified(true);
    }

    private function loadAddress($id)
    {
        $address = new Address($id);
        $address->setThrowExceptionOnError(true);
        return $address;
    }

    private function loadContact($id)
    {
        $contact = new Contact($id);
        $contact->setThrowExceptionOnError(true);
        return $contact;
    }

    /**
     * @return mixed
     */
    public function get_id()
    {
        return $this->id;
    }

    /**
     * @param mixed $id
     * @return ContactAddress
     */
    public function set_id($id)
    {
        $this->id = $id;
        return $this;
    }

    /**
     * @return int
     */
    public function get_contact_id(): int
    {
        return $this->contact_id;
    }

    /**
     * @param int $contact_id
     * @return ContactAddress
     */
    public function set_contact_id(int $contact_id): ContactAddress
    {
        $this->contact_id = $contact_id;
        $this->setIsObjectModified(true);
        return $this;
    }

    /**
     * @return int
     */
    public function get_address_id(): int
    {
        return $this->address_id;
    }

    /**
     * @param int $address_id
     * @return ContactAddress
     */
    public function set_address_id(int $address_id): ContactAddress
    {
        $this->address_id = $address_id;
        $this->setIsObjectModified(true);
        return $this;
    }

    /**
     * @return int
     */
    public function get_priority(): int
    {
        return $this->priority;
    }

    /**
     * @param int $priority
     * @return ContactAddress
     */
    public function set_priority(int $priority): ContactAddress
    {
        $this->priority = $priority;
        $this->setIsObjectModified(true);
        return $this;
    }

    /**
     * @return string
     */
    public function get_type(): string
    {
        return $this->type;
    }

    /**
     * @param string $type
     * @return ContactAddress
     */
    public function set_type(string $type): ContactAddress
    {
        $this->type = $type;
        $this->setIsObjectModified(true);
        return $this;
    }

    /**
     * @return string|null
     */
    public function get_notes(): ?string
    {
        return $this->notes;
    }

    /**
     * @param string $notes
     * @return ContactAddress
     */
    public function set_notes(string $notes): ContactAddress
    {
        $this->notes = $notes;
        $this->setIsObjectModified(true);
        return $this;
    }

    /**
     * @return string
     */
    public function get_status()
    {
        return $this->status;
    }

    /**
     * @param string $status The status of the address
     * @return ContactAddress
     */
    public function set_status($status)
    {
        $this->status = $status == self::STATUS_ACTIVE ? self::STATUS_ACTIVE : self::STATUS_INACTIVE;
        $this->setIsObjectModified(true);
        return $this;
    }

    /**
     * @return string
     */
    public function get_is_primary(): string
    {
        return $this->isPrimary;
    }

    /**
     * @param string $isPrimary
     * @return ContactAddress
     */
    public function set_is_primary($isPrimary): ContactAddress
    {
        $this->isPrimary = $isPrimary == self::IS_PRIMARY_YES ? self::IS_PRIMARY_YES : self::IS_PRIMARY_NO;
        $this->setIsObjectModified(true);
        return $this;
    }

    /**
     * @return Datetime
     */
    public function get_created_date(): Datetime
    {
        return $this->createdDate;
    }

    /**
     * @param Datetime $createdDate
     * @return ContactAddress
     */
    public function set_created_date(Datetime $createdDate): ContactAddress
    {
        $this->createdDate = $createdDate;
        $this->setIsObjectModified(true);
        return $this;
    }

    /**
     * @return Datetime
     */
    public function get_period_start(): Datetime
    {
        return $this->periodStart;
    }

    /**
     * @param Datetime|null $periodStart
     * @return ContactAddress
     */
    public function set_period_start(Datetime $periodStart): ContactAddress
    {
        $this->periodStart = $periodStart;
        $this->setIsObjectModified(true);
        return $this;
    }

    /**
     * @return Datetime|null
     */
    public function get_period_end(): ?Datetime
    {
        return $this->periodEnd;
    }

    /**
     * @param Datetime|null $periodEnd
     * @return ContactAddress
     */
    public function set_period_end(?Datetime $periodEnd): ContactAddress
    {
        $this->periodEnd = $periodEnd;
        $this->setIsObjectModified(true);
        return $this;
    }

    /**
     * @return string
     */
    public function get_author(): string
    {
        return $this->author;
    }

    /**
     * @param string $author
     * @return ContactAddress
     */
    public function set_author(string $author): ContactAddress
    {
        $this->author = $author;
        $this->setIsObjectModified(true);
        return $this;
    }

    /**
     * @return string|null
     */
    public function get_inactivated_reason(): ?string
    {
        return $this->inactivated_reason;
    }

    /**
     * @param string $inactivated_reason
     * @return ContactAddress
     */
    public function set_inactivated_reason(string $inactivated_reason): ContactAddress
    {
        $this->inactivated_reason = $inactivated_reason;
        $this->setIsObjectModified(true);
        return $this;
    }

    /**
     * @return string
     */
    public function get_use(): ?string
    {
        return $this->_use;
    }

    /**
     * @param string $use
     * @return ContactAddress
     */
    public function set_use(?string $use): ContactAddress
    {
        $this->_use = $use;
        $this->setIsObjectModified(true);
        return $this;
    }

    public function toArray()
    {
        return $this->jsonSerialize();
    }

    /**
     * Specify data which should be serialized to JSON
     * @link https://php.net/manual/en/jsonserializable.jsonserialize.php
     * @return mixed data which can be serialized by <b>json_encode</b>,
     * which is a value of any type other than a resource.
     * @since 5.4.0
     */
    public function jsonSerialize()
    {
        $result = [
            "id" => $this->get_id(),
            'use' => $this->get_use(),
            'type' => $this->get_type(),
            'author' => $this->get_author(),
            'period_end' => null,
            'period_start' => null
        ];
        if (!empty($this->get_period_end())) {
            $result['period_end'] = DateFormatterUtils::oeFormatShortDate($this->get_period_end()->format("Y-m-d"));
        }
        if (!empty($this->get_period_start())) {
            $result['period_start'] = DateFormatterUtils::oeFormatShortDate($this->get_period_start()->format("Y-m-d"));
        }
        return $result;
    }
}
